// Hardcoded server URL
// const API_URL = 'http://localhost:8080';
const API_URL = 'https://cigarica.dimitrije.online';

document.addEventListener('DOMContentLoaded', async () => {
  const registerSection = document.getElementById('registerSection');
  const apiKeySection = document.getElementById('apiKeySection');
  const resendSection = document.getElementById('resendSection');
  const mainSection = document.getElementById('mainSection');

  const regEmailInput = document.getElementById('regEmail');
  const regNameInput = document.getElementById('regName');
  const registerBtn = document.getElementById('registerBtn');

  const apiKeyInput = document.getElementById('apiKey');
  const saveApiKeyBtn = document.getElementById('saveApiKey');
  const showRegisterSectionBtn = document.getElementById('showRegisterSection');
  const showRegisterFromLoginBtn = document.getElementById('showRegisterFromLogin');
  const showResendSectionBtn = document.getElementById('showResendSection');
  const backToLoginBtn = document.getElementById('backToLogin');

  const resendEmailInput = document.getElementById('resendEmail');
  const resendActivationBtn = document.getElementById('resendActivationBtn');

  const userNameSpan = document.getElementById('userName');
  const logoutBtn = document.getElementById('logout');
  const goingOutBtn = document.getElementById('goingOutBtn');
  const goingOutScheduledBtn = document.getElementById('goingOutScheduledBtn');
  const cancelBtn = document.getElementById('cancelBtn');
  const activeList = document.getElementById('activeList');
  const scheduledList = document.getElementById('scheduledList');
  const status = document.getElementById('status');

  // Version notification elements
  const versionNotification = document.getElementById('versionNotification');
  const versionMessage = document.getElementById('versionMessage');
  const dismissVersionBtn = document.getElementById('dismissVersion');

  // Group elements
  const groupSelect = document.getElementById('groupSelect');
  const createGroupBtn = document.getElementById('createGroupBtn');
  const joinGroupBtn = document.getElementById('joinGroupBtn');
  const getCodeBtn = document.getElementById('getCodeBtn');
  const viewMembersBtn = document.getElementById('viewMembersBtn');
  const leaveGroupBtn = document.getElementById('leaveGroupBtn');
  const createGroupModal = document.getElementById('createGroupModal');
  const joinGroupModal = document.getElementById('joinGroupModal');
  const inviteCodeModal = document.getElementById('inviteCodeModal');
  const membersModal = document.getElementById('membersModal');
  const newGroupNameInput = document.getElementById('newGroupName');
  const joinGroupIdInput = document.getElementById('joinGroupId');
  const inviteCodeDisplay = document.getElementById('inviteCodeDisplay');
  const membersList = document.getElementById('membersList');
  const confirmCreateGroup = document.getElementById('confirmCreateGroup');
  const cancelCreateGroup = document.getElementById('cancelCreateGroup');
  const confirmJoinGroup = document.getElementById('confirmJoinGroup');
  const cancelJoinGroup = document.getElementById('cancelJoinGroup');
  const copyInviteCode = document.getElementById('copyInviteCode');
  const closeInviteCode = document.getElementById('closeInviteCode');
  const closeMembersModal = document.getElementById('closeMembersModal');

  // Change name elements
  const changeNameBtn = document.getElementById('changeNameBtn');
  const changeNameModal = document.getElementById('changeNameModal');
  const newDisplayName = document.getElementById('newDisplayName');
  const confirmChangeName = document.getElementById('confirmChangeName');
  const cancelChangeName = document.getElementById('cancelChangeName');

  // Store groups data for later use
  let currentGroups = [];

  // Load saved data and determine which section to show
  const data = await chrome.storage.local.get(['cigarica_apiKey', 'cigarica_userName']);

  if (data.cigarica_apiKey && data.cigarica_userName) {
    showMainSection(data.cigarica_userName);
  } else {
    // Default to login page
    await showApiKeySection();
  }

  function showRegisterSection() {
    registerSection.style.display = 'block';
    apiKeySection.style.display = 'none';
    resendSection.style.display = 'none';
    mainSection.style.display = 'none';
  }

  async function showApiKeySection() {
    registerSection.style.display = 'none';
    apiKeySection.style.display = 'block';
    resendSection.style.display = 'none';
    mainSection.style.display = 'none';
  }

  function showResendSection() {
    registerSection.style.display = 'none';
    apiKeySection.style.display = 'none';
    resendSection.style.display = 'block';
    mainSection.style.display = 'none';
  }

  async function showMainSection(userName) {
    registerSection.style.display = 'none';
    apiKeySection.style.display = 'none';
    resendSection.style.display = 'none';
    mainSection.style.display = 'block';
    userNameSpan.textContent = userName;
    await loadGroups();
    loadActiveUsers();
    checkVersion();
  }

  // Version checking
  function compareVersions(v1, v2) {
    const parts1 = v1.split('.').map(Number);
    const parts2 = v2.split('.').map(Number);

    for (let i = 0; i < Math.max(parts1.length, parts2.length); i++) {
      const part1 = parts1[i] || 0;
      const part2 = parts2[i] || 0;

      if (part1 < part2) return -1;
      if (part1 > part2) return 1;
    }
    return 0;
  }

  async function checkVersion() {
    try {
      // Get current version from manifest
      const manifest = chrome.runtime.getManifest();
      const currentVersion = manifest.version;

      // Check if version check was dismissed
      const dismissedData = await chrome.storage.local.get(['cigarica_dismissedVersion']);
      if (dismissedData.cigarica_dismissedVersion === currentVersion) {
        // User already dismissed this version notification
        return;
      }

      // Fetch latest version from server
      const response = await fetch(`${API_URL}/api/version`);
      if (response.ok) {
        const data = await response.json();
        const latestVersion = data.latest_version;

        // Compare versions
        if (compareVersions(currentVersion, latestVersion) < 0) {
          // Current version is older, show notification
          versionMessage.textContent = `New version ${latestVersion} available! Current: ${currentVersion}`;
          versionNotification.style.display = 'flex';
        }
      }
    } catch (err) {
      console.error('Failed to check version:', err);
    }
  }

  // Dismiss version notification
  dismissVersionBtn.addEventListener('click', async () => {
    versionNotification.style.display = 'none';
    const manifest = chrome.runtime.getManifest();
    await chrome.storage.local.set({ cigarica_dismissedVersion: manifest.version });
  });

  async function loadGroups() {
    const data = await chrome.storage.local.get(['cigarica_apiKey', 'cigarica_selectedGroupId']);
    if (!data.cigarica_apiKey) return;

    try {
      const response = await fetch(`${API_URL}/api/groups/list`, {
        headers: { 'Authorization': `Bearer ${data.cigarica_apiKey}` }
      });

      if (response.ok) {
        const result = await response.json();
        const groups = result.groups || [];

        // Store groups data for later use
        currentGroups = groups;

        // Clear and repopulate dropdown
        if (groups.length === 0) {
          groupSelect.innerHTML = '<option value="">No groups - create or join one</option>';
        } else {
          groupSelect.innerHTML = '<option value="">-- Select a group --</option>';
          groups.forEach(group => {
            const option = document.createElement('option');
            option.value = group.id;
            option.textContent = `${group.name} (${group.members.length} members)`;
            groupSelect.appendChild(option);
          });
        }

        // Restore selected group
        if (data.cigarica_selectedGroupId && groups.some(g => g.id === data.cigarica_selectedGroupId)) {
          groupSelect.value = data.cigarica_selectedGroupId;
        } else {
          // Clear invalid selection
          await chrome.storage.local.set({ cigarica_selectedGroupId: '' });
        }

        updateGroupButtons();
      }
    } catch (err) {
      console.error('Failed to load groups:', err);
    }
  }

  function updateGroupButtons() {
    const selectedGroupId = groupSelect.value;
    const hasSelection = selectedGroupId !== '';
    leaveGroupBtn.style.display = hasSelection ? 'block' : 'none';
    getCodeBtn.style.display = hasSelection ? 'block' : 'none';
    viewMembersBtn.style.display = hasSelection ? 'block' : 'none';
  }

  // Register
  registerBtn.addEventListener('click', async () => {
    const email = regEmailInput.value.trim();
    const name = regNameInput.value.trim();

    if (!email || !name) {
      showStatus('Please fill in all fields', 'error');
      return;
    }

    try {
      const response = await fetch(`${API_URL}/api/register`, {
        method: 'POST',
        headers: { 'Content-Type': 'application/json' },
        body: JSON.stringify({ email, name })
      });

      if (response.ok) {
        await chrome.storage.local.set({ cigarica_userName: name });
        showStatus('Check your email for verification!', 'success');
        await showApiKeySection();
      } else {
        const text = await response.text();
        showStatus(text || 'Registration failed', 'error');
      }
    } catch (err) {
      showStatus('Connection error: ' + err.message, 'error');
    }
  });

  // Show register section from login
  showRegisterFromLoginBtn.addEventListener('click', () => {
    showRegisterSection();
  });

  // Show login section from register page
  showRegisterSectionBtn.addEventListener('click', async () => {
    await showApiKeySection();
  });

  // Show resend section from login
  showResendSectionBtn.addEventListener('click', () => {
    showResendSection();
  });

  // Back to login from resend section
  backToLoginBtn.addEventListener('click', async () => {
    await showApiKeySection();
  });

  // Save API Key
  saveApiKeyBtn.addEventListener('click', async () => {
    const apiKey = apiKeyInput.value.trim();

    if (!apiKey) {
      showStatus('Please enter your API key', 'error');
      return;
    }

    try {
      // Validate API key and fetch user info from backend
      const response = await fetch(`${API_URL}/api/user`, {
        method: 'GET',
        headers: { 'Authorization': `Bearer ${apiKey}` }
      });

      if (!response.ok) {
        if (response.status === 401) {
          showStatus('Invalid or unverified API key', 'error');
        } else {
          showStatus('Failed to verify API key', 'error');
        }
        return;
      }

      const userData = await response.json();
      const userName = userData.name;

      // Save and login
      await chrome.storage.local.set({ cigarica_apiKey: apiKey, cigarica_userName: userName });
      showStatus('Logged in!', 'success');
      showMainSection(userName);
    } catch (err) {
      console.error('Login error:', err);
      showStatus('Failed to connect to server', 'error');
    }
  });

  // Resend activation email
  resendActivationBtn.addEventListener('click', async () => {
    const email = resendEmailInput.value.trim();

    if (!email) {
      showStatus('Please enter your email', 'error');
      return;
    }

    try {
      const response = await fetch(`${API_URL}/api/resend-activation`, {
        method: 'POST',
        headers: { 'Content-Type': 'application/json' },
        body: JSON.stringify({ email })
      });

      if (response.ok) {
        showStatus('Activation email sent! Check your inbox.', 'success');
        resendEmailInput.value = '';
        // Go back to login after a delay
        setTimeout(async () => {
          await showApiKeySection();
        }, 2000);
      } else {
        const text = await response.text();
        showStatus(text || 'Failed to resend activation email', 'error');
      }
    } catch (err) {
      showStatus('Connection error: ' + err.message, 'error');
    }
  });

  // Logout
  logoutBtn.addEventListener('click', async () => {
    await chrome.storage.local.clear();
    showStatus('Logged out', 'success');
    showRegisterSection();
  });

  // Group selection change
  groupSelect.addEventListener('change', async () => {
    const groupId = groupSelect.value;
    await chrome.storage.local.set({ cigarica_selectedGroupId: groupId });
    updateGroupButtons();
    loadActiveUsers();
    // Trigger background refresh
    chrome.runtime.sendMessage({ action: 'refresh' });
  });

  // Create group
  createGroupBtn.addEventListener('click', () => {
    createGroupModal.style.display = 'flex';
    newGroupNameInput.value = '';
    newGroupNameInput.focus();
  });

  cancelCreateGroup.addEventListener('click', () => {
    createGroupModal.style.display = 'none';
  });

  confirmCreateGroup.addEventListener('click', async () => {
    const name = newGroupNameInput.value.trim();
    if (!name) {
      showStatus('Please enter a group name', 'error');
      return;
    }

    const data = await chrome.storage.local.get(['cigarica_apiKey']);
    if (!data.cigarica_apiKey) return;

    try {
      const response = await fetch(`${API_URL}/api/groups/create`, {
        method: 'POST',
        headers: {
          'Content-Type': 'application/json',
          'Authorization': `Bearer ${data.cigarica_apiKey}`
        },
        body: JSON.stringify({ name })
      });

      if (response.ok) {
        const result = await response.json();
        createGroupModal.style.display = 'none';
        showStatus(`Group created! ID: ${result.group_id}`, 'success');

        // Copy group ID to clipboard
        await navigator.clipboard.writeText(result.group_id);
        showStatus(`Group created! ID copied to clipboard`, 'success');

        await loadGroups();
        groupSelect.value = result.group_id;
        await chrome.storage.local.set({ cigarica_selectedGroupId: result.group_id });
        updateGroupButtons();
        loadActiveUsers();
      } else {
        const text = await response.text();
        showStatus(text || 'Failed to create group', 'error');
      }
    } catch (err) {
      showStatus('Connection error: ' + err.message, 'error');
    }
  });

  // Join group
  joinGroupBtn.addEventListener('click', () => {
    joinGroupModal.style.display = 'flex';
    joinGroupIdInput.value = '';
    joinGroupIdInput.focus();
  });

  cancelJoinGroup.addEventListener('click', () => {
    joinGroupModal.style.display = 'none';
  });

  confirmJoinGroup.addEventListener('click', async () => {
    const groupId = joinGroupIdInput.value.trim();
    if (!groupId) {
      showStatus('Please enter a group ID', 'error');
      return;
    }

    const data = await chrome.storage.local.get(['cigarica_apiKey']);
    if (!data.cigarica_apiKey) return;

    try {
      const response = await fetch(`${API_URL}/api/groups/join`, {
        method: 'POST',
        headers: {
          'Content-Type': 'application/json',
          'Authorization': `Bearer ${data.cigarica_apiKey}`
        },
        body: JSON.stringify({ group_id: groupId })
      });

      if (response.ok) {
        const result = await response.json();
        joinGroupModal.style.display = 'none';
        showStatus(`Joined "${result.group.name}"!`, 'success');
        await loadGroups();
        groupSelect.value = groupId;
        await chrome.storage.local.set({ cigarica_selectedGroupId: groupId });
        updateGroupButtons();
        loadActiveUsers();
      } else {
        const text = await response.text();
        showStatus(text || 'Failed to join group', 'error');
      }
    } catch (err) {
      showStatus('Connection error: ' + err.message, 'error');
    }
  });

  // Leave group
  leaveGroupBtn.addEventListener('click', async () => {
    const groupId = groupSelect.value;
    if (!groupId) return;

    if (!confirm('Are you sure you want to leave this group?')) {
      return;
    }

    const data = await chrome.storage.local.get(['cigarica_apiKey']);
    if (!data.cigarica_apiKey) return;

    try {
      const response = await fetch(`${API_URL}/api/groups/leave`, {
        method: 'POST',
        headers: {
          'Content-Type': 'application/json',
          'Authorization': `Bearer ${data.cigarica_apiKey}`
        },
        body: JSON.stringify({ group_id: groupId })
      });

      if (response.ok) {
        showStatus('Left group', 'success');
        groupSelect.value = '';
        await chrome.storage.local.set({ cigarica_selectedGroupId: '' });
        await loadGroups();
        updateGroupButtons();
        loadActiveUsers();
      } else {
        const text = await response.text();
        showStatus(text || 'Failed to leave group', 'error');
      }
    } catch (err) {
      showStatus('Connection error: ' + err.message, 'error');
    }
  });

  // Get group invitation code
  getCodeBtn.addEventListener('click', async () => {
    const groupId = groupSelect.value;
    if (!groupId) return;

    const data = await chrome.storage.local.get(['cigarica_apiKey']);
    if (!data.cigarica_apiKey) return;

    try {
      const response = await fetch(`${API_URL}/api/groups/code?group_id=${encodeURIComponent(groupId)}`, {
        headers: { 'Authorization': `Bearer ${data.cigarica_apiKey}` }
      });

      if (response.ok) {
        const result = await response.json();
        inviteCodeDisplay.value = result.invitation_code;
        inviteCodeModal.style.display = 'flex';
      } else {
        const text = await response.text();
        showStatus(text || 'Failed to get invitation code', 'error');
      }
    } catch (err) {
      showStatus('Connection error: ' + err.message, 'error');
    }
  });

  // Copy invitation code
  copyInviteCode.addEventListener('click', async () => {
    try {
      await navigator.clipboard.writeText(inviteCodeDisplay.value);
      showStatus('Invitation code copied to clipboard!', 'success');
      inviteCodeModal.style.display = 'none';
    } catch (err) {
      showStatus('Failed to copy to clipboard', 'error');
    }
  });

  // Close invitation code modal
  closeInviteCode.addEventListener('click', () => {
    inviteCodeModal.style.display = 'none';
  });

  // View group members
  viewMembersBtn.addEventListener('click', () => {
    const selectedGroupId = groupSelect.value;
    if (!selectedGroupId) return;

    const group = currentGroups.find(g => g.id === selectedGroupId);
    if (!group) return;

    // Display members in modal
    if (group.members && group.members.length > 0) {
      membersList.innerHTML = group.members.map(email => `
        <div class="member-item">${email}</div>
      `).join('');
    } else {
      membersList.innerHTML = '<div class="no-members">No members in this group</div>';
    }

    membersModal.style.display = 'flex';
  });

  // Close members modal
  closeMembersModal.addEventListener('click', () => {
    membersModal.style.display = 'none';
  });

  // Change display name
  changeNameBtn.addEventListener('click', async () => {
    const data = await chrome.storage.local.get(['cigarica_userName']);
    newDisplayName.value = data.cigarica_userName || '';
    changeNameModal.style.display = 'flex';
    newDisplayName.focus();
  });

  // Confirm change name
  confirmChangeName.addEventListener('click', async () => {
    const name = newDisplayName.value.trim();

    if (!name) {
      showStatus('Display name cannot be empty', 'error');
      return;
    }

    const data = await chrome.storage.local.get(['cigarica_apiKey']);
    if (!data.cigarica_apiKey) {
      showStatus('Please login first', 'error');
      return;
    }

    try {
      const response = await fetch(`${API_URL}/api/user/name`, {
        method: 'PUT',
        headers: {
          'Content-Type': 'application/json',
          'Authorization': `Bearer ${data.cigarica_apiKey}`
        },
        body: JSON.stringify({ name })
      });

      if (response.ok) {
        await chrome.storage.local.set({ cigarica_userName: name });
        userNameSpan.textContent = name;
        changeNameModal.style.display = 'none';
        showStatus('Display name updated!', 'success');
      } else {
        const text = await response.text();
        showStatus(text || 'Failed to update name', 'error');
      }
    } catch (err) {
      console.error('Update name error:', err);
      showStatus('Failed to connect to server', 'error');
    }
  });

  // Cancel change name
  cancelChangeName.addEventListener('click', () => {
    changeNameModal.style.display = 'none';
  });

  // Going out - always 10 minutes
  goingOutBtn.addEventListener('click', () => {
    goSmoke(10, 0);
  });

  // Going out scheduled - in 10 minutes
  goingOutScheduledBtn.addEventListener('click', () => {
    goSmoke(10, 10);
  });

  // Cancel / I'm back
  cancelBtn.addEventListener('click', async () => {
    await cancelSmoke();
  });

  async function goSmoke(minutes, delayMinutes = 0) {
    const data = await chrome.storage.local.get(['cigarica_apiKey', 'cigarica_selectedGroupId']);

    if (!data.cigarica_apiKey) {
      showStatus('Please register first', 'error');
      return;
    }

    const groupId = data.cigarica_selectedGroupId || '';

    // Require group selection
    if (!groupId) {
      showStatus('Please select or create a group first', 'error');
      return;
    }

    try {
      const response = await fetch(`${API_URL}/api/activate`, {
        method: 'POST',
        headers: {
          'Content-Type': 'application/json',
          'Authorization': `Bearer ${data.cigarica_apiKey}`
        },
        body: JSON.stringify({ minutes, delay_minutes: delayMinutes, group_id: groupId })
      });

      if (response.ok) {
        const groupName = groupSelect.options[groupSelect.selectedIndex].text;
        if (delayMinutes > 0) {
          showStatus(`Scheduled break in ${delayMinutes} min (${minutes} min duration) in ${groupName}!`, 'success');
        } else {
          showStatus(`Smoke break started (${minutes} min) in ${groupName}!`, 'success');
        }
        loadActiveUsers();
        // Trigger background refresh
        chrome.runtime.sendMessage({ action: 'refresh' });
      } else {
        const text = await response.text();
        showStatus(text || 'Failed to activate', 'error');
      }
    } catch (err) {
      showStatus('Connection error: ' + err.message, 'error');
    }
  }

  async function cancelSmoke() {
    const data = await chrome.storage.local.get(['cigarica_apiKey', 'cigarica_selectedGroupId']);

    if (!data.cigarica_apiKey) {
      showStatus('Please register first', 'error');
      return;
    }

    const groupId = data.cigarica_selectedGroupId || '';

    // Require group selection
    if (!groupId) {
      showStatus('Please select a group first', 'error');
      return;
    }

    try {
      const response = await fetch(`${API_URL}/api/deactivate`, {
        method: 'POST',
        headers: {
          'Content-Type': 'application/json',
          'Authorization': `Bearer ${data.cigarica_apiKey}`
        },
        body: JSON.stringify({ group_id: groupId })
      });

      if (response.ok) {
        showStatus('Smoke break cancelled!', 'success');
        loadActiveUsers();
        // Trigger background refresh
        chrome.runtime.sendMessage({ action: 'refresh' });
      } else {
        const text = await response.text();
        showStatus(text || 'Failed to cancel', 'error');
      }
    } catch (err) {
      showStatus('Connection error: ' + err.message, 'error');
    }
  }

  function showStatus(message, type) {
    status.textContent = message;
    status.className = type;
    setTimeout(() => {
      status.textContent = '';
      status.className = '';
    }, 3000);
  }

  async function loadActiveUsers() {
    try {
      const storageData = await chrome.storage.local.get(['cigarica_selectedGroupId', 'cigarica_userName']);
      const groupId = storageData.cigarica_selectedGroupId || '';
      const myName = storageData.cigarica_userName || '';

      const url = groupId
        ? `${API_URL}/api/active?group_id=${encodeURIComponent(groupId)}`
        : `${API_URL}/api/active`;

      const response = await fetch(url);
      const data = await response.json();

      // Check if current user has an active or scheduled break
      const iAmActive = data.active && data.active.some(user => user.name === myName);
      const iAmScheduled = data.scheduled && data.scheduled.some(user => user.name === myName);

      // Show/hide cancel button based on whether user has a break
      if (iAmActive || iAmScheduled) {
        cancelBtn.style.display = 'block';
      } else {
        cancelBtn.style.display = 'none';
      }

      // Display active users
      if (data.active && data.active.length > 0) {
        activeList.innerHTML = data.active.map(user => {
          const expiresAt = new Date(user.expires_at);
          const remaining = Math.round((expiresAt - new Date()) / 60000);
          return `<div class="user">${user.name} <span class="time">(${remaining}min left)</span></div>`;
        }).join('');
      } else {
        activeList.innerHTML = '<div class="empty">No one is smoking right now</div>';
      }

      // Display scheduled users
      if (data.scheduled && data.scheduled.length > 0) {
        scheduledList.innerHTML = data.scheduled.map(user => {
          const scheduledStart = new Date(user.scheduled_start);
          const minutesUntil = Math.round((scheduledStart - new Date()) / 60000);
          return `<div class="user">${user.name} <span class="time">(in ${minutesUntil}min)</span></div>`;
        }).join('');
      } else {
        scheduledList.innerHTML = '<div class="empty">No scheduled breaks</div>';
      }
    } catch (err) {
      activeList.innerHTML = '<div class="error">Could not load active users</div>';
      scheduledList.innerHTML = '<div class="error">Could not load scheduled users</div>';
    }
  }
});
