// Hardcoded server URL
// const API_URL = 'http://localhost:8080';
const API_URL = 'https://cigarica.dimitrije.online';

// Initialize alarm on install
chrome.runtime.onInstalled.addListener(() => {
  chrome.alarms.create('checkActive', { periodInMinutes: 1 });
  checkActiveUsers();
});

// Also check on startup
chrome.runtime.onStartup.addListener(() => {
  chrome.alarms.create('checkActive', { periodInMinutes: 1 });
  checkActiveUsers();
});

// Handle alarm
chrome.alarms.onAlarm.addListener((alarm) => {
  if (alarm.name === 'checkActive') {
    checkActiveUsers();
  }
});

async function checkActiveUsers() {
  try {
    const data = await chrome.storage.local.get(['cigarica_userName', 'cigarica_selectedGroupId', 'cigarica_notifiedSessions']);
    const myName = data.cigarica_userName || '';
    const groupId = data.cigarica_selectedGroupId || '';
    const notifiedSessions = data.cigarica_notifiedSessions || {};

    const url = groupId
      ? `${API_URL}/api/active?group_id=${encodeURIComponent(groupId)}`
      : `${API_URL}/api/active`;

    const response = await fetch(url);
    const result = await response.json();

    const activeUsers = result.active || [];
    const activeCount = activeUsers.length;

    // Update badge with only active count (not scheduled)
    if (activeCount > 0) {
      chrome.action.setBadgeText({ text: activeCount.toString() });
      chrome.action.setBadgeBackgroundColor({ color: '#FF6B35' });
    } else {
      chrome.action.setBadgeText({ text: '' });
    }

    // Check for new smokers and notify
    const currentActiveSessions = new Set();

    for (const user of activeUsers) {
      // Create unique session ID using name + expires_at timestamp
      const sessionId = `${user.name}:${user.expires_at}`;
      currentActiveSessions.add(sessionId);

      // Only notify if we haven't notified about this specific session and it's not the current user
      if (!notifiedSessions[sessionId] && user.name !== myName) {
        // New smoker detected, show notification
        chrome.notifications.create({
          type: 'basic',
          iconUrl: 'icon128.png',
          title: 'Cigarica',
          message: `${user.name} is going for a smoke!`,
          priority: 1
        });

        // Mark this session as notified
        notifiedSessions[sessionId] = Date.now();
      }
    }

    // Clean up old sessions (remove entries older than 24 hours)
    const oneDayAgo = Date.now() - (24 * 60 * 60 * 1000);
    for (const sessionId in notifiedSessions) {
      if (notifiedSessions[sessionId] < oneDayAgo) {
        delete notifiedSessions[sessionId];
      }
    }

    // Also remove sessions that are no longer active to keep storage clean
    for (const sessionId in notifiedSessions) {
      const expiresAt = sessionId.split(':')[1];
      if (expiresAt && new Date(expiresAt) < new Date()) {
        delete notifiedSessions[sessionId];
      }
    }

    // Save updated notified sessions
    await chrome.storage.local.set({ cigarica_notifiedSessions: notifiedSessions });

  } catch (err) {
    console.error('Failed to check active users:', err);
    chrome.action.setBadgeText({ text: '?' });
    chrome.action.setBadgeBackgroundColor({ color: '#999' });
  }
}

// Also allow manual refresh from popup
chrome.runtime.onMessage.addListener((message, sender, sendResponse) => {
  if (message.action === 'refresh') {
    checkActiveUsers();
    sendResponse({ success: true });
  }
});
